package com.burhanstore.neonbubblequest.Utils

import android.annotation.SuppressLint
import android.app.Activity
import android.app.WallpaperManager
import android.content.Context
import android.graphics.BitmapFactory
import android.net.ConnectivityManager
import android.net.NetworkCapabilities
import android.net.Uri
import android.os.Build
import android.os.StatFs
import android.provider.Settings
import android.telephony.TelephonyManager
import android.view.View
import android.view.inputmethod.InputMethodManager
import android.widget.Toast
import androidx.appcompat.app.AppCompatActivity
import androidx.browser.customtabs.CustomTabsIntent
import com.google.firebase.messaging.FirebaseMessaging
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch
import org.json.JSONObject
import java.io.InputStream
import java.net.HttpURLConnection
import java.net.URL
import java.text.SimpleDateFormat
import java.util.Locale
import java.util.TimeZone


object AppString {

    fun getCountry(context: Context): String {
        return try {
            val teleMgr = context.getSystemService(Context.TELEPHONY_SERVICE) as TelephonyManager
            val countryIso = teleMgr.networkCountryIso
            if (countryIso.isNullOrEmpty()) "Unknown" else Locale("", countryIso).displayCountry
        } catch (e: Exception) {
            e.printStackTrace()
            "Unknown"
        }
    }

    fun generateUid(): String {
        val chars = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789"
        return (1..100)
            .map { chars.random() }
            .joinToString("")
    }


    fun generateReferId(): String {
        val chars = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789"
        return (1..10)
            .map { chars.random() }
            .joinToString("")
    }

    @SuppressLint("HardwareIds")
    fun getAndroidId(context: Context): String {
        return try {
            val androidId = Settings.Secure.getString(
                context.contentResolver,
                Settings.Secure.ANDROID_ID
            )
            androidId ?: "Unknown"
        } catch (e: Exception) {
            e.printStackTrace()
            "Unknown"
        }
    }

    // Function to get FCM token
    fun getFcmToken(onTokenReceived: (String) -> Unit) {
        FirebaseMessaging.getInstance().token
            .addOnCompleteListener { task ->
                if (!task.isSuccessful) {
                    onTokenReceived("") // Return empty string if failed
                    return@addOnCompleteListener
                }
                val token = task.result ?: ""
                onTokenReceived(token)
            }
    }

    fun setWallpaper(context: Context, imageUrl: String, type: String) {

        DialogManager.showDialog()
        CoroutineScope(Dispatchers.IO).launch {
            try {
                val url = URL(imageUrl)
                val connection: HttpURLConnection = url.openConnection() as HttpURLConnection
                connection.doInput = true
                connection.connect()
                val inputStream: InputStream = connection.inputStream
                val bitmap = BitmapFactory.decodeStream(inputStream)

                val wallpaperManager = WallpaperManager.getInstance(context)
                when (type) {
                    "home" -> {
                        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.N) {
                            wallpaperManager.setBitmap(
                                bitmap,
                                null,
                                true,
                                WallpaperManager.FLAG_SYSTEM
                            )
                        } else {
                            wallpaperManager.setBitmap(bitmap)
                        }
                    }

                    "lock" -> {
                        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.N) {
                            wallpaperManager.setBitmap(
                                bitmap,
                                null,
                                true,
                                WallpaperManager.FLAG_LOCK
                            )
                        } else {
                            Toast.makeText(
                                context,
                                "Lock screen wallpaper requires Android N+",
                                Toast.LENGTH_SHORT
                            ).show()
                        }
                    }

                    "both" -> {
                        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.N) {
                            wallpaperManager.setBitmap(
                                bitmap,
                                null,
                                true,
                                WallpaperManager.FLAG_SYSTEM or WallpaperManager.FLAG_LOCK
                            )
                        } else {
                            wallpaperManager.setBitmap(bitmap)
                        }
                    }
                }

                launch(Dispatchers.Main) {
                    DialogManager.dismissDialog()
                    Toast.makeText(context, "Wallpaper set successfully!", Toast.LENGTH_SHORT)
                        .show()
                }

            } catch (e: Exception) {
                e.printStackTrace()
                launch(Dispatchers.Main) {
                    DialogManager.dismissDialog()
                    Toast.makeText(context, "Failed to set wallpaper", Toast.LENGTH_SHORT).show()
                }
            }
        }
    }

    class JsonReader(json: String) {

        private var jsonObject: JSONObject? = null

        init {
            try {
                jsonObject = JSONObject(json)
            } catch (e: Exception) {
                e.printStackTrace()
            }
        }

        fun get(key: String): String {
            return jsonObject?.optString(key, "") ?: ""
        }
    }


    fun isInternetConnected(activity: AppCompatActivity): Boolean {
        val connectivityManager =
            activity.getSystemService(Context.CONNECTIVITY_SERVICE) as ConnectivityManager
        val network = connectivityManager.activeNetwork ?: return false
        val networkCapabilities =
            connectivityManager.getNetworkCapabilities(network) ?: return false

        return networkCapabilities.hasCapability(NetworkCapabilities.NET_CAPABILITY_INTERNET) &&
                networkCapabilities.hasCapability(NetworkCapabilities.NET_CAPABILITY_VALIDATED)
    }

    fun openCustomChrome(url: String, activity: Activity) {
        val builder = CustomTabsIntent.Builder()
        val customTabsIntent = builder.build()
        customTabsIntent.launchUrl(activity, Uri.parse(url))
    }

    fun getMB(activity: Activity): String {
        val path = activity.filesDir
        val stat = StatFs(path.path)
        val blockSize = stat.blockSizeLong
        val totalBlocks = stat.blockCountLong
        val availableBlocks = stat.availableBlocksLong

        val totalSizeBytes = blockSize * totalBlocks
        val availableSizeBytes = blockSize * availableBlocks
        val usedSizeBytes = totalSizeBytes - availableSizeBytes
        val usedSizeMB = usedSizeBytes / (1024 * 1024)

        return usedSizeMB.toString()
    }

    fun extractYT(url: String): String? {
        val uri = Uri.parse(url)
        return uri.getQueryParameter("v")
    }


    fun formatDate(dateStr: String): String {
        return try {
            val inputFormat = SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss.SSSSSS'Z'", Locale.ENGLISH)
            inputFormat.timeZone = TimeZone.getTimeZone("UTC") // API time is in UTC

            val outputFormat = SimpleDateFormat("d MMM yyyy h:mm a", Locale.ENGLISH)
            outputFormat.timeZone = TimeZone.getDefault() // convert to local time

            val date = inputFormat.parse(dateStr)
            outputFormat.format(date ?: return "Invalid Date")
        } catch (e: Exception) {
            e.printStackTrace()
            "Invalid Date"
        }
    }

    fun hideKeyboard(activity: Activity) {
        val imm = activity.getSystemService(Activity.INPUT_METHOD_SERVICE) as InputMethodManager
        // Find the currently focused view, so we can grab the correct window token from it
        var view = activity.currentFocus
        if (view == null) {
            view = View(activity)
        }
        imm.hideSoftInputFromWindow(view.windowToken, 0)
    }



}
